unit u_logging;

// Purpose: Handles all logging

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

procedure AddToLog (LogString: shortstring);
procedure AddToEventLog (EventMessage: shortstring);
procedure AddToPmsLog (LogString: shortstring);
procedure LogIDs;
procedure LogData;
procedure LogPmsRaw;
procedure WriteParametersToEventLog;
procedure NewLogFileNames;

var
  ExeDir, LoggingPath, LogFileName, EventLogFileName, PmsLogFilename, LogStr: shortstring;

implementation

uses u_main, u_messages, u_calc, u_timestrings;

procedure AddToLog (LogString: shortstring);
var
  FileName: shortstring;
  LogFile: text;
begin
  Filename:= LogFileName;
  assign (logFile, Filename);
  if FileExists (FileName) then append (LogFile) else rewrite (LogFile);
  writeln (LogFile, DateTimeStamp + LogString);
  close (LogFile);
end;

procedure AddToEventLog (EventMessage: shortstring);
var
  FileName: shortstring;
  EventFile: text;
begin
  FileName:= EventLogFileName;
  assign (EventFile, FileName);
  if FileExists (FileName) then append (EventFile) else rewrite (EventFile);
  writeln (EventFile, DateTimeStamp + ': ' + EventMessage);
  close (EventFile);
end;

procedure AddToPmsLog (LogString: shortstring);
var
  FileName: shortstring;
  LogFile: text;
begin
  Filename:= PmsLogFileName;
  assign (logFile, Filename);
  if FileExists (FileName) then append (LogFile) else rewrite (LogFile);
  writeln (LogFile, DateTimeStamp + LogString);
  close (LogFile);
end;

procedure LogIDs;
begin
  LogStr:= '**,Status,AHHC_C,APC_C,ABC_C, WC_C,TPC_C1,TPC_C1,AMC_M,APC_M,TPC_M1,TPC_M2,AMR_M,APR_1,APR_2, TPR_1, TPR_2,TPR2_M, Minor, Major, Angle, QAmpX, QAmpY, QAngX, QAngY,Temp,Hygr,  Baro,';
  AddToLog (LogStr);
end;

procedure LogData;
// Because of problems with locale and comma separated files we do not
// log floating point numbers. In stead integers with the values 10, 100, or 1000 fold.
begin
  LogStr:=
    format (' ,%8x, ', [Status]) +    // Status in Hex format
    format ('%4d, ', [HalfHeightCenter_Cap]) +
    format ('%4d, ', [APeakCenter_Cap]) +
    format ('%4d, ', [ABaseCenter_Cap]) +
    format ('%4d, ', [WidthCenter_Cap]) +
    format ('%5d, ', [TPassCenter_Cap_HFSW1]) +
    format ('%5d, ', [TPassCenter_Cap_HFSW2]) +

    format ('%4d, ', [AMidCenter_Mag]) +
    format ('%4d, ', [APeakCenter_Mag]) +
    format ('%5d, ', [TPassCenter_Mag_HFSW1]) +
    format ('%5d, ', [TPassCenter_Mag_HFSW2]) +

    format ('%4d, ', [AMidRim_Mag]) +
    format ('%4d, ', [APeakRim1_Mag_HFSW1]) +
    format ('%4d, ', [APeakRim1_Mag_HFSW2]) +
    format ('%5d, ', [TPassRim1_Mag_HFSW1]) +
    format ('%5d, ', [TPassRim1_Mag_HFSW2]) +
    format ('%5d, ', [TPassRim2_Mag]) +

    format ('%5d, ', [round (EllipseMinorAxis_mm)]) +
    format ('%5d, ', [round (EllipseMajorAxis_mm)]) +
    format ('%5d, ', [round (PrecessionAngle_Degrees)]) +
    format ('%5d, ', [round (QAmplitudeX * 10000)]) +
    format ('%5d, ', [round (QAmplitudeY * 10000)]) +
    format ('%5d, ', [round (QAngleX * 180 / pi)]) +
    format ('%5d, ', [round (QAngleY * 180 / pi)]) +
    format ('%3d, ', [BME_Temperature]) +
    format ('%3d, ', [BME_Hygro]) +
    format ('%5d, ', [BME_Baro]);
  ;
  if MadeCompassScreenShot then LogStr += 'ScreenShot,';
  MadeCompassScreenShot:= false;
  AddToLog (LogStr);
end;

procedure LogPmsRaw;
var PmsStatus: byte;
begin
  PmsStatus:= 0;
  if HalfSwing then PmsStatus += $01;
  if SeenCenter_Mag then PmsStatus += $40;
  if SeenCenter_Cap then PmsStatus += $80;
  LogStr:= format (' *, %4d, %5d, %4d, %4d, %4d, %4d',
    [PmsStatus, PositionCounter, Adc_North, Adc_South, Adc_East, Adc_West]);
  AddToPmsLog (LogStr);
end;

procedure WriteParametersToEventLog;
begin
  AddToEventLog ('Changed Settings');
  AddToEventLog        ('Version GUI:      ' + VERSIONSTRING);
  AddToEventLog (format('Version Firmware:       %6d',[FirmwareversionNumber]));

  AddToEventLog (format('DriveSyncMode:               %1d',[ord (Drive_SyncMode)]));
  AddToEventLog (format('Drive_Position:         %6d',[Drive_Position]));
  AddToEventLog (format('Drive_Width:              %4d',[Drive_Width]));
  AddToEventLog (format('Drive_MaximalCurrent:     %4d',[Drive_MaximalCurrent]));
  AddToEventLog (format('Drive_MinimalCurrent:     %4d',[Drive_MinimalCurrent]));

  AddToEventLog (format('En Det Center_Mag:           %1d',[ord (EnableDetectorCenterPass_Mag)]));
  AddToEventLog (format('En Det Center_Cap:           %1d',[ord (EnableDetectorCenterPass_Cap)]));

  AddToEventLog (format('TStartLookCenter_Mag:   %6d',[TStartLookForCenter_Mag]));
  AddToEventLog (format('TMissedCenter_Mag:      %6d',[TMissedCenter_Mag]));
  AddToEventLog (format('TStartLookCenter_Cap:   %6d',[TStartLookForCenter_Cap]));
  AddToEventLog (format('TMissedCenter_Cap:      %6d',[TMissedCenter_Cap]));

  AddToEventLog (format('RimSyncMode:                 %1d',[ord (RimSyncMode)]));
  AddToEventLog (format('TStartLookRim1_Mag:     %6d',[TStartLookForRim1_Mag]));
  AddToEventLog (format('TMissedRim1_Mag:        %6d',[TMissedRim1_Mag]));
  AddToEventLog (format('TStartLookRim2_Mag:     %6d',[TStartLookForRim2_Mag]));
  AddToEventLog (format('TMissedRim2_Mag:        %6d',[TMissedRim2_Mag]));

  AddToEventLog (format('AmplitudeControlMode:        %1d',[ord (AmplitudeControlMode)]));
  AddToEventLog (format('SetPoint_Amplitude_mm:    %4d',[SetPoint_Amplitude_Ticks]));
  AddToEventLog (format('RimCoilRadius_mm:         %4d',[RimCoilRadius_mm]));
  AddToEventLog (format('CenterElectrodeRadius_mm:   %5.2f',[CenterElectrodeRadius_mm]));
  AddToEventLog (format('CenterTarget_Cap_ticks:   %4d',[CenterTarget_Cap_ticks]));

  AddToEventLog (format('DDS_XtalFrequency:%12d',[DDS_XtalFrequency]));
  AddToEventLog (format('DDS_FrequencyWord:%12d',[DDS_FrequencyWord]));
  AddToEventLog (format('Divider_T5:              %5d',[Divider_T5]));
  AddToEventLog (format('Divider_Final:           %5d',[Divider_Final]));

  AddToEventLog (format('PMSZeroAveragefactor:       %5.2f',[PMSZeroAveragefactor]));
  AddToEventLog (format('CompassScale:            %5d',[CompassScale]));
  AddToEventLog ('****************************************');
end;

procedure NewLogFileNames;
begin
  LogFileName:= LoggingPath + DateTimeStamp2;
  FrmMain.TbxLogFileName.text:= LogFileName;
  EventLogFileName:= LogFileName + '.event';
  PmsLogFilename:=   LogFileName + '.pms';
  LogFileName:=      LogFileName + '.log';
  AddToLog      ('**' + TITLE + VERSIONSTRING);
  AddToEventLog ('**' + TITLE + VERSIONSTRING);
  AddToPmsLog   ('**' + TITLE + VERSIONSTRING);
  LogIDs;
end;

begin
// no unit initialisation code
end.

