 /////////////////////////////////////////////////////////////////////////////////////////
// Module Name: Messages.cpp
// Project: Foucault Pendulum, Subsystem BobControl
// Target CPU: Arduino MEGA + ETHernet Shield
// Target hardware: BobControl_v1
// Last Modification: See below
// Author: Jan Breemer, jan@breem.nl
// www.foucaultpendulum.nl
/////////////////////////////////////////////////////////////////////////////////////////

#include "Globals.h"
#include <Ethernet.h>
#include <EthernetUdp.h>

// For Ethernet communication
EthernetUDP Udp;
byte mac[] = {0x90, 0xA2, 0xDA, 0x0F, 0x67, 51};
IPAddress ArduinoIP (192, 168, 2, 51);
unsigned int ArduinoPort = 8888;

// For InMessage:
#define INMESSAGESIZE 40
byte InMessageBuffer[INMESSAGESIZE];
tDrive_SyncMode Drive_SyncMode, PrevDrive_SyncMode;
tRimSyncMode RimSyncMode;
tAmplitudeControlMode AmplitudeControlMode;
unsigned long Command, PrevCommand, DDS_FrequencyWord;
unsigned int  
  TStartLookForCenter_Mag, TMissedCenter_Mag,
  TStartLookForCenter_Cap, TMissedCenter_Cap,
  TStartLookForRim1_Mag, TMissedRim1_Mag,
  TStartLookForRim2_Mag, TMissedRim2_Mag,
  SetPoint_Amplitude_Ticks,
  TDrive_Start, TDrive_Stop, 
  Drive_MinimalCurrent, Drive_MaximalCurrent,
  Divider_T5, Divider_Final;
bool
  EnableDetectorCenterPass_Mag, EnableDetectorCenterPass_Cap, EnableDetectorRimPass_Mag,
  HaveComm, EnableDrive, 
  ForceReSync, ForceMinimalDriveCurrent, ForceMaximalDriveCurrent,
  DoUpdate_DDS, DoWriteParametersToEEprom, DoClearResetCause;  
  
// for OutMessage:
#define OUTMESSAGESIZE 75
byte OutMessageBuffer[OUTMESSAGESIZE + 1];  // payload + 1 extra for length byte
unsigned long Status;
byte MessageNumber;  // to check for lost messages.

// For clarity we list all functions in this module in the order they appear
void DecodeInMessage (void);
void PrepareOutMessage (void);
void Update_Messages (void);
void Init_Messages (void);

// ********************************************************************************************************
void DecodeInMessage (void) // takes ca  usec
{
  DIAGPIN_A8_H
  // Copy values from messagebuffer 
  // Byte numbering is identical to Pascal
  memcpy (&Command,                  &InMessageBuffer[ 0], 4);
  memcpy (&TStartLookForCenter_Mag,  &InMessageBuffer[ 4], 2);
  memcpy (&TMissedCenter_Mag,        &InMessageBuffer[ 6], 2);
  memcpy (&TStartLookForCenter_Cap,  &InMessageBuffer[ 8], 2);
  memcpy (&TMissedCenter_Cap,        &InMessageBuffer[10], 2);
  memcpy (&TStartLookForRim1_Mag,    &InMessageBuffer[12], 2);
  memcpy (&TMissedRim1_Mag,          &InMessageBuffer[14], 2);
  memcpy (&TStartLookForRim2_Mag,    &InMessageBuffer[16], 2);
  memcpy (&TMissedRim2_Mag,          &InMessageBuffer[18], 2);
  memcpy (&SetPoint_Amplitude_Ticks, &InMessageBuffer[20], 2);
  memcpy (&TDrive_Start,             &InMessageBuffer[22], 2);
  memcpy (&TDrive_Stop,              &InMessageBuffer[24], 2);
  memcpy (&Drive_MinimalCurrent,     &InMessageBuffer[26], 2);
  memcpy (&Drive_MaximalCurrent,     &InMessageBuffer[28], 2);
  memcpy (&DDS_FrequencyWord,        &InMessageBuffer[30], 4);
  memcpy (&Divider_T5,               &InMessageBuffer[34], 2);
  memcpy (&Divider_Final,            &InMessageBuffer[36], 2);     

  // Process values
 
  Drive_SyncMode =                (Command & 0x00000003); // 0 Mag, 1 Cap, 2 Charron, 3 Resonance
  if                             ((Command & 0x00000004) == 0x00000008) DidReSyncMyself = false;
  if                             ((Command & 0x00000008) == 0x00000008) ForceReSync = true;

  RimSyncMode =                   (Command & 0x00000030) >> 4;  
  // bit 6 not used
  if                             ((Command & 0x00000080) == 0x00000080) DoUpdate_DDS = true;

  AmplitudeControlMode =          (Command & 0x00000300) >> 8;  // 0:Rim1_Mag 1:Center_Cap 2:TBD 3: None
  // bit 10 not used
  if                             ((Command & 0x00000800) == 0x00000800) HalfSwing = !HalfSwing; // Invert HalfSwing 

  EnableDetectorCenterPass_Mag = ((Command & 0x00001000) == 0x00001000);
  EnableDetectorCenterPass_Cap = ((Command & 0x00002000) == 0x00002000);
  EnableDetectorRimPass_Mag    = ((Command & 0x00004000) == 0x00004000);
  // bit 15 not used
  
  // bit 16 .. 23 not used

  ForceMaximalDriveCurrent =     ((Command & 0x01000000) == 0x01000000);
  ForceMinimalDriveCurrent =     ((Command & 0x02000000) == 0x02000000);
  EnableDrive =                  ((Command & 0x04000000) == 0x04000000);
  // bit 27 not used
 
  if                             ((Command & 0x10000000) == 0x10000000) DoWriteParametersToEEprom = true;
  // bit 29 not used
  // bit 30 not used
  if                             ((Command & 0x80000000) == 0x80000000) ResetArduino ();

  if (Command != PrevCommand) 
  {
    sprintf (PrintStr, "Command changed: %8lx", Command);
    Serial.println (PrintStr);
    PrevCommand = Command;
  }
  DIAGPIN_A8_L
}

// ********************************************************************************************************
void PrepareOutMessage (void) // takes ca.  usec
{
  DIAGPIN_A8_H
  // Prepare values
  OutMessageBuffer[0] = OUTMESSAGESIZE; // string length on the Pascal side
  
  Status = 0;
  if (SeenCenter_Mag)           Status |= 0x00000001; SeenCenter_Mag   = false; // only once
  if (MissedCenter_Mag)         Status |= 0x00000002; MissedCenter_Mag = false; // only once
  if (SeenCenter_Cap)           Status |= 0x00000004; SeenCenter_Cap   = false; // only once
  if (MissedCenter_Cap)         Status |= 0x00000008; MissedCenter_Cap = false; // only once
 
  if (SeenRim1_Mag)             Status |= 0x00000010; SeenRim1_Mag   = false; // only once
  if (MissedRim1_Mag)           Status |= 0x00000020; MissedRim1_Mag = false; // only once
  if (SeenRim2_Mag)             Status |= 0x00000040; SeenRim2_Mag   = false; // only once
  if (MissedRim2_Mag)           Status |= 0x00000080; MissedRim2_Mag = false; // only once

  if (HalfSwing)                Status |= 0x00000100;
  if (HaveSync)                 Status |= 0x00000200; 
  // bit 10 spare
  if (Touch_Charron)            Status |= 0x00000800; // no futher implementation
 
  if (UsingMaximalDriveCurrent) Status |= 0x00001000;
  if (UsingMinimalDriveCurrent) Status |= 0x00002000;
  // bit 14 spare
  if (DDS_FrequencyChanged)     Status |= 0x00008000; // by Synchronisation Control in resonance Mode
  DDS_FrequencyChanged = false;

  if (OverRangeCenter_Mag)      Status |= 0x00010000; OverRangeCenter_Mag = false; // only once
  if (OverRangeCenter_Cap)      Status |= 0x00020000; OverRangeCenter_Cap = false; // only once
  if (OverRangeRim1_Mag)        Status |= 0x00040000; OverRangeRim1_Mag   = false; // only once
  if (OverRangeRim2_Mag)        Status |= 0x00080000; OverRangeRim2_Mag   = false; // only once
  
  // 2 nibbles spare
  
  // bit 28 spare
  if (DidReSyncMyself)          Status |= 0x20000000;
  if (!BME280_IsInitialized)    Status |= 0x40000000; 
  if (GeneralError)             Status |= 0x80000000; 
  
  // Copy values to the messagebuffer
  // byte numbering is identical to Pascal side
  memcpy (&OutMessageBuffer[1],  &FirmwareVersionNumber, 2);
  memcpy (&OutMessageBuffer[3],  &Status, 4);
  
  // Center detection Capacitive
  memcpy (&OutMessageBuffer[7],  &Adc_Center_Cap, 2);
  memcpy (&OutMessageBuffer[9],  &APeakCenter_Cap, 2);
  memcpy (&OutMessageBuffer[11], &ABaseCenter_Cap, 2);
  memcpy (&OutMessageBuffer[13], &TPassCenter_Cap, 2);
  memcpy (&OutMessageBuffer[15], &AHalfHeightCenter_Cap, 2);
  memcpy (&OutMessageBuffer[17], &WidthCenter_Cap, 2);

  // Center detection Magnetic
  memcpy (&OutMessageBuffer[19], &Adc_Center_Mag, 2);
  memcpy (&OutMessageBuffer[21], &APeakCenter_Mag, 2);
  memcpy (&OutMessageBuffer[23], &AMidCenter_Mag, 2);
  memcpy (&OutMessageBuffer[25], &TPassCenter_Mag, 2);

  // Rim detection Capacitive
  memcpy (&OutMessageBuffer[27], &Adc_Rim_Cap, 2); // currently not further supported
  
  // Rim detection Magnetic
  memcpy (&OutMessageBuffer[29], &Adc_Rim_Mag, 2);
  memcpy (&OutMessageBuffer[31], &AMidRim_Mag, 2);
  memcpy (&OutMessageBuffer[33], &APeakRim1_Mag, 2);
  memcpy (&OutMessageBuffer[35], &TPassRim1_Mag, 2);
  memcpy (&OutMessageBuffer[37], &APeakRim2_Mag, 2);
  memcpy (&OutMessageBuffer[39], &TPassRim2_Mag, 2);

  // PMS data at sending message
  memcpy (&OutMessageBuffer[41], &Adc_North, 2);
  memcpy (&OutMessageBuffer[43], &Adc_South, 2);
  memcpy (&OutMessageBuffer[45], &Adc_East, 2);
  memcpy (&OutMessageBuffer[47], &Adc_West, 2);
  if (Drive_SyncMode == Drive_SyncByCenter_Mag) memcpy (&OutMessageBuffer[49], &PositionCounter_Mag, 2);
  if (Drive_SyncMode == Drive_SyncByCenter_Cap) memcpy (&OutMessageBuffer[49], &PositionCounter_Cap, 2);
  
  // PMS data frozen at Center Pass
  memcpy (&OutMessageBuffer[51], &Center_North, 2);
  memcpy (&OutMessageBuffer[53], &Center_South, 2);
  memcpy (&OutMessageBuffer[55], &Center_East, 2);
  memcpy (&OutMessageBuffer[57], &Center_West, 2);

  // For Resonance Drive
  memcpy (&OutMessageBuffer[59], &DDS_FrequencyWord, 4);
  memcpy (&OutMessageBuffer[63], &TResonanceDrive, 2);

  // Environment
  memcpy (&OutMessageBuffer[65], &BME_Temperature, 2);
  memcpy (&OutMessageBuffer[67], &BME_Baro, 2);
  OutMessageBuffer[69] = BME_Hygro;

  // Check for lost messages
  OutMessageBuffer[70] = MessageNumber++;
  DIAGPIN_A8_L
}

//********************************************************************************************************
void Update_Messages (void) // called frequently from loop ()
{
  static byte CommLedCounter;
  
  unsigned int PacketSize = Udp.parsePacket ();
  if (PacketSize == INMESSAGESIZE)
  {
    //Serial.println ("Msg Rcv");
    if (++CommLedCounter > 10) {COMMLED_TOGGLE; CommLedCounter = 0;}
    HaveComm = true;
    Udp.read (InMessageBuffer, INMESSAGESIZE); 
    DecodeInMessage ();
    PrepareOutMessage ();
    Udp.beginPacket (Udp.remoteIP (), Udp.remotePort ());
    Udp.write (&OutMessageBuffer[0], OUTMESSAGESIZE + 1); // one larger than the payload
    Udp.endPacket ();
  } 
}

//********************************************************************************************************
void Init_Messages (void) // called at startup from setup()
{
  Serial.println ("\nInitializing UDP Communication.");
  Serial.print ("Arduino IP =  ");
  Serial.print (ArduinoIP);
  Serial.print (" Port = ");
  Serial.println (ArduinoPort);
 
  Ethernet.begin (mac, ArduinoIP);
  bool success = Udp.begin (ArduinoPort);
  Serial.print ("UDP setup: ");
  if (success) Serial.println ("OK"); else Serial.println ("Failed");
}
 
