/////////////////////////////////////////////////////////////////////////////////////////
// Module Name: FW_Pendulum.ino
// Project: Foucault Pendulum, Subsystem BobControl
// Target CPU: Arduino MEGA + ETHernet Shield
// Target hardware: PCB BobControl_v1
// Last Modification: See below
// Author: Jan Breemer, jan@breem.nl
// www.foucaultpendulum.nl
/////////////////////////////////////////////////////////////////////////////////////////

#include "Globals.h"
#define PROJECTNAME "FW_Pendulum"

// Fill in the firmware timestamp below. 
#define BUILD_YY 25
#define BUILD_MM 11
#define BUILD_DD 03
unsigned int FirmwareVersionNumber; // Will be composed in setup() and goes into the GUI Message.
char PrintStr[100]; // for diagnostic messages
bool 
  DoReportParameters, GeneralError,
  Opt1, Opt2, Opt3, Opt4;

// for BME280 sensor for Temperature, Hygro an Barometric pressure
#include <Adafruit_BME280.h>
Adafruit_BME280 BME280;
bool BME280_IsInitialized;
int BME_Temperature, BME_Hygro, BME_Baro;

// For clarity we list all functions in this module
// in the order they appear
void ReadOptionJumpers (void);
void SendWordToDDS (unsigned int Word);
void SendFrequencyWordToDDS (void);
void Init_DDS (void);
ISR (TIMER5_COMPA_vect);
void TestLeds (void);
byte EEPROM_ReadByte (unsigned int EEpromAddress);
unsigned long EEPROM_ReadLong (unsigned int EEpromAddress);
void EEPROM_WriteByte (unsigned int EEpromAddress, byte EEpromData);
void EEPROM_WriteInt (unsigned int EEpromAddress, int EEpromData);
void EEPROM_WriteLong (unsigned int EEpromAddress, unsigned long EEpromData);
void EEPROM_PARAMETERS_Clear (void);
void WriteParametersToEEprom (void);
void ReadParametersFromEEprom (void);
void ReportParameters (void);
void ReadBME (void);
void ResetArduino (byte ResetCause);
void ShowHelp (void); 
void ListAdcValues (void);
void setup (void);
void loop (void);

//**************************************************************************************************/
void ReadOptionJumpers (void)
{
  // Jumper placed gives low = true
  Opt1 = (PINC & 0x40) == 0; // PC6, 1-2 jumpered
  Opt2 = (PINC & 0x10) == 0; // PC4, 3-4 jumpered
  Opt3 = (PINC & 0x01) == 0; // PC2, 5-6 jumpered
  Opt4 = (PINC & 0x04) == 0; // PC0, 7-8 jumpered
}

//**************************************************************************************************/
void SendWordToDDS (unsigned int Word) 
{
  DDS_FSYNC_HIGH;
  DDS_SCLK_HIGH;
  DDS_FSYNC_LOW;
  for (byte i=0; i<16; i++) // we do 16 bits
  {
    if ((Word & 0x8000) == 0) DDS_SDATA_LOW else DDS_SDATA_HIGH; // MSB first
    DDS_SCLK_LOW; // this latches the data bit
    Word = Word << 1; // shift left one bit
    DDS_SCLK_HIGH; 
  }
  DDS_FSYNC_HIGH; // ends the transfer
}

//**************************************************************************************************/
void SendFrequencyWordToDDS (void)
{
  DoUpdate_DDS = false;
  // The AD9833 needs the FrequencyWord to be sent in two parts of 14 bits.
  // We assume the rightmost 28 bits of Fword to be the data.
  // We put the data in 2 longints.
  unsigned int Part1, Part2;
  Part1 = Part2 = 0x4000; // we write to Frequency word 0
  Part1 |=  (DDS_FrequencyWord & 0x00003FFF);        // copy bits 13.. 0 in positions 13..0
  Part2 |= ((DDS_FrequencyWord & 0x0FFFC000) >> 14); // copy bits 27..14 in positions 13..0 
  // the control word disables the reset, 
  // enables the mode for two-step loading of the frequency word,
  // disables the DAC output,
  // feeds DAC-MSB bit div 2 to the output.
  sprintf (PrintStr, "Set DDS FreqWord dec: %8ld hex: %4x %4x", DDS_FrequencyWord, Part2, Part1); 
  Serial.println (PrintStr);
 
  SendWordToDDS (0x2028); // control word 
  SendWordToDDS (Part1); // first the 14 LSB's of Freqword in a 16 bit u_int
  SendWordToDDS (Part2); //  then the 14 MSB's of Freqword in a 16 bit u_int
  DDS_FrequencyChanged = true;
}

//**************************************************************************************************/
void Init_DDS (void)
{
  DDRA |= 0x07; // make DDS pins output
  SendWordToDDS (0x2128); // reset on
  DDS_FrequencyWord = 4992899; // initially 465 kHz
  SendFrequencyWordToDDS ();
  
  // Configure 16 bit Timer 5 for external clock input from DDS, 465 kHz
  TCCR5A = 0x00; // WaveformMode 4 = CTC
  TCCR5B = 0x0F; // External clock input T5 pin 47, WaveformMode 4 = CTC.
  TIMSK5 = 0x02; // Interrupt on OCRA match
  OCR5A = 23250; // Divider to start with
}

//********************************************************************************************************
ISR (TIMER5_COMPA_vect) 
// This Timer is clocked by the DDS frequency of 465 kHz and forms Divider_T5.
{
  static unsigned int CounterFinal;
  DIAGPIN_A9_T 
  // We do the reload of the timer in the interrupt, where the timer is always at zero, 
  // to prevent the situation where the timer has already passed the new value. 
  // In that case it would have to go through the whole 16 bit range first. 
  OCR5A = Divider_T5;
  if (CounterFinal > 0)  
    CounterFinal--;
  else
  {
    CounterFinal = Divider_Final; // reset it
    if (Drive_SyncMode == Drive_SyncByResonance)
      PositionCounter_Drive = TDrive_Start - 10; // allow CenterDetect_Mag signal gating 
    DIAGPIN_A10_T 
    //Serial.println ("Drive Pulse for Resonance");
  }  
  // This part has been tested so far (not with a pendulum) and works.
  // But to really use it there should also be a tracking 
  // mechanism to fine adjust the frequency.
  // At the moment of writing I don't yet know how to do that.
}

/**************************************************************************************************/
void TestLeds (void)
{
  for (byte i=0; i<3; i++)
  {
    CENTERLED_ON     delay(200);
    DRIVELED_ON      delay(200);
    RIMLED_1_ON      delay(200);
    RIMLED_2_ON      delay(200);
    HALFSWINGLED_ON  delay(200);
    COMMLED_ON       delay(200);
    SPARELED_1_ON    delay(200);
    SPARELED_2_ON    delay(200);

    CENTERLED_OFF    delay(200);
    DRIVELED_OFF     delay(200);
    RIMLED_1_OFF     delay(200);
    RIMLED_2_OFF     delay(200);
    HALFSWINGLED_OFF delay(200);
    COMMLED_OFF      delay(200);
    SPARELED_1_OFF   delay(200);
    SPARELED_2_OFF   delay(200); 
  }
}

/**************************************************************************************************/
byte EEPROM_ReadByte (unsigned int EEpromAddress)
{
  // Wait for completion of previous action
  while (EECR & (1 << EEPE));
  EEAR = EEpromAddress;
  EECR |= (1 << EERE);
  return EEDR;
}

/**************************************************************************************************/
unsigned int EEPROM_ReadInt (unsigned int EEpromAddress)
{
  int lo, hi;
  lo = EEPROM_ReadByte (EEpromAddress);
  hi = EEPROM_ReadByte (EEpromAddress + 1);
  return ((hi << 8) | lo);
}

/**************************************************************************************************/
unsigned long EEPROM_ReadLong (unsigned int EEpromAddress)
{
  long b3, b2, b1, b0;
  b0 = EEPROM_ReadByte (EEpromAddress);
  b1 = EEPROM_ReadByte (EEpromAddress + 1);
  b2 = EEPROM_ReadByte (EEpromAddress + 2);
  b3 = EEPROM_ReadByte (EEpromAddress + 3);
  return (b3 << 24 | b2 << 16 | b1 << 8 | b0);
}

/**************************************************************************************************/
void EEPROM_WriteByte (unsigned int EEpromAddress, byte EEpromData)
{
  // Wait for completion of previous action
  while (EECR & (1 << EEPE));
  EEAR = EEpromAddress;
  EEDR = EEpromData;
  EECR |= (1 << EEMPE);
  EECR |= (1 << EEPE);
}

/**************************************************************************************************/
void EEPROM_WriteInt (unsigned int EEpromAddress, int EEpromData)
{   
  EEPROM_WriteByte (EEpromAddress,      lowByte (EEpromData));
  EEPROM_WriteByte (EEpromAddress + 1, highByte (EEpromData));
}

//**************************************************************************************************/
void EEPROM_WriteLong (unsigned int EEpromAddress, unsigned long EEpromData)
{
  EEPROM_WriteByte (EEpromAddress, EEpromData & 0x000000FF);
  EEPROM_WriteByte (EEpromAddress + 1, (EEpromData >> 8)  & 0x000000FF);
  EEPROM_WriteByte (EEpromAddress + 2, (EEpromData >> 16) & 0x000000FF);
  EEPROM_WriteByte (EEpromAddress + 3, (EEpromData >> 24) & 0x000000FF);
}

/**************************************************************************************************/
void Erase_EEPROM_Area (void)
{
  Serial.println ("Erasing Parameters from EEPROM");
  for (byte i=0; i<40; i++) 
    EEPROM_WriteByte (EEPROM_STARTADDRESS_PARAMETERS + i, 255); // looks as never written to
  Serial.println ("Erasing done/n");
}

/**************************************************************************************************/
void WriteParametersToEEprom (void)
{
  byte DetectorsUsed;
  DoWriteParametersToEEprom = false;
  DetectorsUsed = 0;
  DetectorsUsed |= EnableDetectorCenterPass_Mag; 
  DetectorsUsed |= EnableDetectorCenterPass_Cap << 1;
  DetectorsUsed |= EnableDetectorRimPass_Mag << 2;

  EEPROM_WriteByte (EEPROM_STARTADDRESS_PARAMETERS     , DetectorsUsed);
  EEPROM_WriteByte (EEPROM_STARTADDRESS_PARAMETERS +  1, Drive_SyncMode);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS +  2, TStartLookForCenter_Mag);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS +  4, TMissedCenter_Mag);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS +  6, TStartLookForCenter_Cap);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS +  8, TMissedCenter_Cap);

  EEPROM_WriteByte (EEPROM_STARTADDRESS_PARAMETERS + 10, RimSyncMode);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 11, TStartLookForRim1_Mag);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 13, TMissedRim1_Mag);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 15, TStartLookForRim2_Mag);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 17, TMissedRim2_Mag);

  EEPROM_WriteByte (EEPROM_STARTADDRESS_PARAMETERS + 19, AmplitudeControlMode);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 20, SetPoint_Amplitude_Ticks);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 22, TDrive_Start);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 24, TDrive_Stop);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 26, Drive_MinimalCurrent);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 28, Drive_MaximalCurrent);
  
  EEPROM_WriteLong (EEPROM_STARTADDRESS_PARAMETERS + 30, DDS_FrequencyWord);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 34, Divider_T5);
  EEPROM_WriteInt  (EEPROM_STARTADDRESS_PARAMETERS + 36, Divider_Final);
  Serial.println ("Done Writing to EEPROM");
  DoReportParameters = true;
  Init_DDS ();
}

/**************************************************************************************************/
void ReadParametersFromEEprom (void)
{
  byte DetectorsUsed      =  EEPROM_ReadByte (EEPROM_STARTADDRESS_PARAMETERS);
  EnableDetectorCenterPass_Mag = (DetectorsUsed & 0x01) == 0x01; 
  EnableDetectorCenterPass_Cap = (DetectorsUsed & 0x02) == 0x02;
  EnableDetectorRimPass_Mag    = (DetectorsUsed & 0x04) == 0x04;
  
  Drive_SyncMode =           EEPROM_ReadByte (EEPROM_STARTADDRESS_PARAMETERS +  1);
  TStartLookForCenter_Mag =  EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS +  2);
  TMissedCenter_Mag =        EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS +  4);
  TStartLookForCenter_Cap =  EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS +  6);
  TMissedCenter_Cap =        EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS +  8);

  RimSyncMode =              EEPROM_ReadByte (EEPROM_STARTADDRESS_PARAMETERS + 10);
  TStartLookForRim1_Mag =    EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 11);
  TMissedRim1_Mag =          EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 13);
  TStartLookForRim2_Mag =    EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 15);
  TMissedRim2_Mag =          EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 17);

  AmplitudeControlMode =     EEPROM_ReadByte (EEPROM_STARTADDRESS_PARAMETERS + 19);
  SetPoint_Amplitude_Ticks = EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 20); 
  TDrive_Start =             EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 22);
  TDrive_Stop =              EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 24);
  Drive_MinimalCurrent =     EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 26);
  Drive_MaximalCurrent =     EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 28);
  
  DDS_FrequencyWord =        EEPROM_ReadLong (EEPROM_STARTADDRESS_PARAMETERS + 30);
  Divider_T5 =               EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 34);
  Divider_Final =            EEPROM_ReadInt  (EEPROM_STARTADDRESS_PARAMETERS + 36);
  Serial.println ("Done Reading from EEPROM");
  SendFrequencyWordToDDS ();
  DoReportParameters = true;
}

//********************************************************************************************************
void ReportParameters (void) // on Change Parameters and entering letter "p"
{
  DoReportParameters = false; // at Change Parameters
  Serial.println ("\nList of current Parameter Settings:");
  Serial.print ("Detector Center_Magetic    ");
  if (EnableDetectorCenterPass_Mag) Serial.println ("is used");
  else Serial.println ("is not used");
  Serial.print ("Detector Center_Capacitive ");
  if (EnableDetectorCenterPass_Cap) Serial.println ("is used");
  else  Serial.println ("is not used");
  Serial.print ("Detector Rim Magnetic      ");
  if (EnableDetectorRimPass_Mag) Serial.println ("is used\n");
  else Serial.println ("is not used");
  
  Serial.println ("\nSettings for Drive pulse:");
  Serial.print ("TDrive_Start             "); Serial.println (TDrive_Start); 
  Serial.print ("TDrive_Stop              "); Serial.println (TDrive_Stop);  
  Serial.print ("Drive_MinimalCurrent     "); Serial.println (Drive_MinimalCurrent); 
  Serial.print ("Drive_MaximalCurrent     "); Serial.println (Drive_MaximalCurrent);  

  Serial.println ("\nSettings for synchronisation:");
  Serial.print ("Drive_SyncMode           "); Serial.println (Drive_SyncMode);
  Serial.print ("TStartLookForCenter_Mag  "); Serial.println (TStartLookForCenter_Mag);
  Serial.print ("TMissedCenter_Mag        "); Serial.println (TMissedCenter_Mag);  
  Serial.print ("TStartLookForCenter_Cap  "); Serial.println (TStartLookForCenter_Cap);
  Serial.print ("TMissedCenter_Cap        "); Serial.println (TMissedCenter_Cap); 
  
  Serial.println ("\nSettings for Rim detection:");
  Serial.print ("RimSyncMode              "); Serial.println (RimSyncMode);
  Serial.print ("TStartLookForRim1_Mag    "); Serial.println (TStartLookForRim1_Mag);
  Serial.print ("TMissedRim1_Mag          "); Serial.println (TMissedRim1_Mag); 
  Serial.print ("TStartLookForRim2_Mag    "); Serial.println (TStartLookForRim2_Mag);
  Serial.print ("TMissedRim2_Mag          "); Serial.println (TMissedRim2_Mag);   

  Serial.println ("\nSettings for Automatic Amplitude Control:");
  Serial.print ("AmplitudeControlMode     "); Serial.println (AmplitudeControlMode);
  Serial.print ("Setpoint Amplitude Ticks "); Serial.println (SetPoint_Amplitude_Ticks);
   
  Serial.println ("\nSettings for Resonance Major Axis:");
  Serial.print ("DDS_FrequencyWord        "); Serial.println (DDS_FrequencyWord);  
  Serial.print ("Divider_T5               "); Serial.println (Divider_T5); 
  Serial.print ("Divider_Final            "); Serial.println (Divider_Final);  
  
  Serial.println ("\nSettings of Option Jumpers:");
  ReadOptionJumpers ();
  Serial.print ("Opt1 (no Led Test       )"); if (Opt1) Serial.println ("jumpered = true"); else Serial.println ("empty = false");
  Serial.print ("Opt2 (Show States       )"); if (Opt2) Serial.println ("jumpered = true"); else Serial.println ("empty = false");
  Serial.print ("Opt3 (no Width Detection)"); if (Opt3) Serial.println ("jumpered = true"); else Serial.println ("empty = false");
  Serial.print ("Opt4 (not assigned)      "); if (Opt4) Serial.println ("jumpered = true"); else Serial.println ("empty = false");
     
  Serial.println ();
}

//********************************************************************************************************
void ReadBME (void)
{
  DoReadBME = false;
  if (BME280_IsInitialized)
  {
    BME_Temperature = round (BME280.readTemperature () * 10); // deci degrees C
    BME_Hygro =       round (BME280.readHumidity ());         // % relative
    BME_Baro =        round (BME280.readPressure () / 10);    // decioPascal
  }
  else BME_Temperature = BME_Hygro = BME_Baro = 0;
   
  if (true) // diagnose
  {
    char Str[40];
    sprintf (Str, "Read BME280: T=%4d Hy=%3d Ba=:%4d", BME_Temperature, BME_Hygro, BME_Baro);
   // Serial.println (Str);
  }
 }

//********************************************************************************************************
void ResetArduino (void)
{
  // Resetting the Arduino from software can be done by jumping to the reset vector, but
  // then the hardware will not be reset, particularly the Ethernet Shield.
  // Here we pull the RESET line LOW with a Fet, causing a complete hardware reset.
  // This requires a special circuit to stretch and invert the reset pulse. See diagram.
  DDRH |= 0x20;  // Data direction = Output
  PORTH |= 0x20; // PORTH bit 5 pin 8 high brings the FET in conduction after some delay.
}

//********************************************************************************************************
void ShowHelp (void) 
{
  Serial.println ("\nAvailable commands: \n");
  Serial.println ("a: Show actual ADC values");
  Serial.println ("E: Erase EEPROM Area");
  Serial.println ("H: Show this Help info");
  Serial.println ("l: Test Leds");
  Serial.println ("p: Show Parameters");
  Serial.println ("P: Read Parameters from EEPROM and show"); 
  Serial.println ("R: RESET ARDUINO");
  Serial.println ("W: Write parameters to EEProm");
}

//********************************************************************************************************
void ListAdcValues (void)
{
  sprintf (PrintStr, "%4d %4d %4d %4d   %4d %4d %4d %4d",
  Adc_North, Adc_South, Adc_West, Adc_East, 
  Adc_Center_Cap, Adc_Center_Mag, Adc_Rim_Cap, Adc_Rim_Mag);
  Serial.println (PrintStr);   
}

//********************************************************************************************************
void setup (void)
{
  Serial.begin(19200); 
  Serial.println ("\n\n\n\n\n\n\n\n"); // make some space
 
  // we compose the FW version as YMMDD, where Y can go from 0 to 6 before 
  // we have an integer overflow. So can be used upto 2031
  FirmwareVersionNumber = (BUILD_YY - 25) * 10000 + BUILD_MM * 100 + BUILD_DD;
  sprintf (PrintStr, "%s Version %6ld coded %5d" , 
  PROJECTNAME, FirmwareVersionNumber + 250000, FirmwareVersionNumber); 
  Serial.println (PrintStr);
   
  Wire.begin (); // initialize the I2C system for the BME280 in the Top Unit.
  
  // I/O's
  DDRA |= 0xFF;  // DDS control
  DDRC |= 0xAA;  // OPT jumper inputs, LED outputs
  PORTC |= 0x55; // Activate Internal Pullup's for Jumper Block OPT
  DDRG  |= 0x01; // !Drive pulse
  PORTG |= 0x01; // !Drive pulse starts in OFF state = high
  DDRL  |= 0xBB; // Leds, Amplifier Gate, except PL2 = DDS input

  // Configure 16 bit Timer 4 for 10 bit Fast PWM (~16 kHz) on channel A, pin 6
  TCCR4A = 0x83;  // Waveform mode 7, Clear on Compare Match
  TCCR4B = 0x09;  // Internal clock, prescaler 1, WaveformMode 7
  OCR4A = 512;    // PWM value To start with   
  DDRH |= 0x08;   // Activate OCR4A, Enable port H3 on pin 6,

  ReadParametersFromEEprom ();
  ReportParameters ();
  
  if (!Opt1) TestLeds (); else Serial.println ("Skipping Led Test"); // jumper OPT1 disables Led Test

  Init_DDS ();
  Init_BobControl ();
  Init_Messages ();

  Serial.print ("Initialize BME280 sensor for Temperature, Hygro and Baro:....");
  BME280_IsInitialized = true;
  if (!BME280.begin ())
  {
    Serial.println ("\nCould not find a valid BME280 sensor, check wiring!\n");
    BME280_IsInitialized = false;
  }
  else  Serial.println ("OK\n");
  DoReadBME = true;

  USEDIAGPIN_A8  // Message in / out
  USEDIAGPIN_A9  // Timer T5
  USEDIAGPIN_A10 // Trigger Resonance Drive Method
  USEDIAGPIN_A11 // Twidth of CenterPass Capacitive
  USEDIAGPIN_A12 // Toggle on CenterPass Capacitive
  USEDIAGPIN_A13 // High on Rim1-, Low on Rim2 Pass Magnetic
  USEDIAGPIN_A14 // Toggle on CenterPass Magnetic
  USEDIAGPIN_A15 // 20 kHz Timer 1 interrupt handler InService time
}

// ********************************************************************************************************
void loop (void)
{
  Update_Messages (); 
 
  char SerIn[2];
  if (Serial.available ())
  {
    Serial.readBytes (SerIn, 2);
    if (SerIn[0] == 'a') ListAdcValues ();
    if (SerIn[0] == 'E') Erase_EEPROM_Area ();
    if (SerIn[0] == 'h') ShowHelp ();
    if (SerIn[0] == 'l') TestLeds ();
    if (SerIn[0] == 'P') ReadParametersFromEEprom ();
    if (SerIn[0] == 'p') DoReportParameters = true;
    if (SerIn[0] == 'R') ResetArduino (); 
    if (SerIn[0] == 'W') WriteParametersToEEprom ();  
  } 

  // DoReportParameters is set at startup and by entering 'p' on the serial monitor
  if (DoReportParameters) ReportParameters();

  // DoWriteParameterToEEEprom is set when changing parameters
  if (DoWriteParametersToEEprom) WriteParametersToEEprom ();

  // DoReadBME is set by OneMinuteCounter in BobCtl / Timer interrupt.
  if (DoReadBME) ReadBME ();

  // DoUpdate_DDS is set by Command in message from GUI
  if (DoUpdate_DDS) SendFrequencyWordToDDS ();
   
  if (false)  // diagnose A/D conversions
  {
    char Str[60];
    sprintf (Str, "Read BME: %4d %4d %4d %4d   %4d %4d",
      Adc_West, Adc_East, Adc_North, Adc_South, Adc_Center_Cap, Adc_Center_Mag);
    Serial.println (Str);
    delay (100);
  }  
}
