 /////////////////////////////////////////////////////////////////////////////////////////
// Module Name: BobControl.cpp
// Project: Foucault Pendulum, Subsystem BobControl
// Target CPU: Arduino MEGA + ETHernet Shield
// Target hardware: BobControl_v1
// Last Modification: See below
// Author: Jan Breemer, jan@breem.nl
// www.foucaultpendulum.nl
/////////////////////////////////////////////////////////////////////////////////////////

#include "Globals.h"
 
bool 
  HaveSync, HalfSwing, 
  DidResetMyself, DidReSyncMyself,
  SeenCenter_Cap, MissedCenter_Cap, 
  SeenCenter_Mag, MissedCenter_Mag,
  OverRangeCenter_Cap, OverRangeCenter_Mag,
  SeenRim1_Mag, MissedRim1_Mag,
  SeenRim2_Mag, MissedRim2_Mag,
  OverRangeRim1_Mag, OverRangeRim2_Mag,
  MaxDrive, UsingMinimalDriveCurrent, UsingMaximalDriveCurrent,
  DDS_FrequencyChanged, DoReadBME, Touch_Charron;
unsigned int 
  PositionCounter_Mag, PositionCounter_Cap, PositionCounter_Rim,
  PositionCounter_Drive, // counters for all timing.
  Adc_North, Adc_South, Adc_East, Adc_West, 
  Adc_Center_Cap, Adc_Center_Mag,
  Adc_Rim_Cap, Adc_Rim_Mag, 
  Center_North, Center_South, Center_East, Center_West,
  APeakCenter_Cap, ABaseCenter_Cap, 
  TPassCenter_Cap, AHalfHeightCenter_Cap, 
  WidthCenter_Cap,
  AMidCenter_Mag, APeakCenter_Mag, 
  TPassCenter_Mag,
  AMidRim_Mag,
  APeakRim1_Mag, TPassRim1_Mag,
  APeakRim2_Mag, TPassRim2_Mag,
  TResonanceDrive,
  DriveLedCounter,
  SeenCenterLedCounter;

// For clarity we list all functions in this module
// in the order they appear

void DetectCenter_Cap (void);
void DetectCenter_Mag (void);
void DetectRim_Mag (void); 
ISR (TIMER1_COMPA_vect);
void Init_BobControl (void);

//********************************************************************************************************
// Statemachine to detect Center Passes Capacitive
// Called at 20 kHz rate from Timer1 interrupt handler
void DetectCenter_Cap (void)
{
  enum tStateCenter_Cap {ccap_Idle = 0, ccap_DontLook, ccap_WaitT1,
                         ccap_TrackRising, ccap_TrackFalling,
                         ccap_WaitSampleBaseLevel, ccap_SampleBaseLevel};
                      
  static tStateCenter_Cap stCenter_Cap;
  static long avWidthCenter_Cap;
  static unsigned int Track, avBaseLevel, avPeakLevel, TQuarterSwing, PulseWidthCounter;
  static byte BaseLevelSamples;

  bool ShowStates =         Opt2; // for diagnose
  bool SkipWidthDetection = Opt3; // for diagnose
  
  PositionCounter_Cap++;
  PulseWidthCounter++;
  
  if (ForceReSync) 
    if (Drive_SyncMode == Drive_SyncByCenter_Cap) // prepare for new Center Pass search 
    {
      ForceReSync = false;
      stCenter_Cap = ccap_Idle;
      Serial.println ("Force Resync");
    }

  if (PositionCounter_Cap > TMissedCenter_Cap)
  {
    MissedCenter_Cap = true;
    if (Drive_SyncMode == Drive_SyncByCenter_Cap) 
      if (Drive_SyncMode == Drive_SyncByCenter_Cap) PositionCounter_Drive = 0;
    Serial.print ("Missed Center_Cap in state:"); Serial.println (stCenter_Cap); 
    stCenter_Cap = ccap_Idle; 
  }
    
  switch (stCenter_Cap) // Statemachine to detect Center Passes Capacitive
  {
    case ccap_Idle:                    // state 0
      AHalfHeightCenter_Cap = 400; // initial value, will be augmented
      avPeakLevel = 400; 
      PositionCounter_Cap = 0;
      if (Drive_SyncMode == Drive_SyncByCenter_Cap) HaveSync = false;
      stCenter_Cap = ccap_DontLook;
      if (ShowStates) Serial.println (stCenter_Cap); // 1
    break;

    case ccap_DontLook:                // state 1
      if (PositionCounter_Cap > TStartLookForCenter_Cap)
      {
        Track = 0;
        stCenter_Cap = ccap_WaitT1;     
        if (SkipWidthDetection) stCenter_Cap = ccap_TrackRising;
        if (ShowStates) Serial.println (stCenter_Cap);  // 2, 3
      }  
    break;

    case ccap_WaitT1:                  // state 2
      if (Adc_Center_Cap > Track) Track = Adc_Center_Cap;
      if (Track > AHalfHeightCenter_Cap)  // we are at T1
      { 
        PulseWidthCounter = 0; 
        //Serial.print ("PT1 "); Serial.println (PositionCounter_Cap);
        //Serial.print ("HT1 "); Serial.println (AHalfHeightCenter_Cap);
        DIAGPIN_A11_H
        stCenter_Cap = ccap_TrackRising;
        if (ShowStates) Serial.println (stCenter_Cap); // 3
      }  
 
    case ccap_TrackRising:             // state 3
      if (Adc_Center_Cap > Track) Track = Adc_Center_Cap; 
      if (Adc_Center_Cap < Track - 5) // we are over the top
      {
        SeenCenter_Cap = true;
        TPassCenter_Cap = PositionCounter_Cap;
        APeakCenter_Cap = Track;
        //Serial.print ("ACenter_Cap "); Serial.println (APeakCenter_Cap);
        //Serial.print ("TPassCenter "); Serial.println (TPassCenter_Cap);
        TQuarterSwing = PositionCounter_Cap >> 1; // div 2 
        if (Drive_SyncMode == Drive_SyncByCenter_Cap)
        {
          PositionCounter_Cap = 0;
          if (RimSyncMode == RimSyncByCenter_Cap) PositionCounter_Rim = 0;
          PositionCounter_Drive = 0;
          Center_North = Adc_North; // freeze PMS data at CenterPass
          Center_South = Adc_South;
          Center_West  = Adc_West;
          Center_East  = Adc_East;
          HaveSync = true;
          HalfSwing = !HalfSwing;
          SeenCenterLedCounter = 2000;
          RIMLED_1_OFF
          RIMLED_2_OFF 
        }
        Track = 1023; // prepare for track falling
        DIAGPIN_A12_T        
        stCenter_Cap = ccap_TrackFalling;
        if (SkipWidthDetection) stCenter_Cap = ccap_WaitSampleBaseLevel;
        if (ShowStates) Serial.println (stCenter_Cap); // 4, 5
      }
    break;

    case ccap_TrackFalling:                // state 4
      if (Adc_Center_Cap < Track) Track = Adc_Center_Cap;
      if (Track < AHalfHeightCenter_Cap)  // we are at T2
      {
        WidthCenter_Cap = PulseWidthCounter;
        if (AmplitudeControlMode == AmplitudeByCenter_Cap)
        {
          // Leaking Bucket Average
          avWidthCenter_Cap = avWidthCenter_Cap - (avWidthCenter_Cap >> 4) + WidthCenter_Cap;
          MaxDrive = ((avWidthCenter_Cap >> 4) > SetPoint_Amplitude_Ticks);
          sprintf (PrintStr, "Av Width: %5d Drive: ", avWidthCenter_Cap >> 4);
          Serial.print (PrintStr); Serial.println (MaxDrive ? "Max" : "Min");
        }  
        //Serial.print ("FP "); Serial.println (PositionCounter_Cap);
        //Serial.print ("HH "); Serial.println (AHalfHeightCenter_Cap);
        //Serial.print ("WC "); Serial.println (WidthCenter_Cap);
        DIAGPIN_A11_L
        stCenter_Cap = ccap_WaitSampleBaseLevel;
        if (ShowStates) Serial.println (stCenter_Cap); // 5
      }    
    break;

    case ccap_WaitSampleBaseLevel:        // state 5
      if (PositionCounter_Cap > TQuarterSwing)
      {
        BaseLevelSamples = 0;
        stCenter_Cap = ccap_SampleBaseLevel;
        if (ShowStates) Serial.println (stCenter_Cap); // 6
      }
    break;
    
    case ccap_SampleBaseLevel:             // state 6
      // apply leaking bucket averaging with factor 1/16
      avBaseLevel = avBaseLevel - (avBaseLevel >> 4) + Adc_Center_Cap;
      ABaseCenter_Cap = avBaseLevel >> 4;
      if (++BaseLevelSamples >= 20) // average 20 samples
      {
        // recalculate HalfHeightLevel
        if (avPeakLevel == 0) avPeakLevel = APeakCenter_Cap << 4; // first time
          else // apply leaking bucket averaging with factor 1/16
            avPeakLevel = avPeakLevel - (avPeakLevel >> 4) + APeakCenter_Cap;  
        unsigned int AveragedAPeakLevel = avPeakLevel >> 4;
        AHalfHeightCenter_Cap = (AveragedAPeakLevel + ABaseCenter_Cap) >> 1; // div 2
        //AHalfHeightCenter_Cap = 401;
        //Serial.print ("ABB "); Serial.println (ABaseCenter_Cap);
        //Serial.print ("APP "); Serial.println (AveragedAPeakLevel);
        //Serial.print ("AHH "); Serial.println (AHalfHeightCenter_Cap);
        stCenter_Cap = ccap_DontLook;  // start over
        if (ShowStates) Serial.println (stCenter_Cap); // 1
      }  
    break;
  }
}

//********************************************************************************************************
// Statemachine to detect Center Passes from Center Coil or Drive Coil
// Called at 20 kHz rate from Timer1 interrupt handler
void DetectCenter_Mag (void) 
{
  enum tStateCenter_Mag {cmag_Idle = 0, cmag_DontLook, 
                         cmag_TrackRising, cmag_WaitCenter,  
                         cmag_WaitSampleMidCenter, cmag_SampleMidCenter};
 
  static tStateCenter_Mag stCenter_Mag;
  static int Track, TQuarterSwing, avMidCenter;
  static byte MidCenterSamples;
  
  bool ShowStates = Opt2; // for diagnose
  
  PositionCounter_Mag++;
  
  if (ForceReSync) 
    if (Drive_SyncMode == Drive_SyncByCenter_Mag) // prepare for new Center Pass search 
    {
      ForceReSync = false;
      stCenter_Mag = cmag_Idle;
      Serial.println ("Force Resync");
    }

  if (PositionCounter_Mag > TMissedCenter_Mag)
  {
    MissedCenter_Mag = true;
    if (Drive_SyncMode == Drive_SyncByCenter_Mag) PositionCounter_Drive = 0;
    Serial.print ("Missed Center_Mag in state: "); Serial.println (stCenter_Mag);
    stCenter_Mag = cmag_Idle;
  }

  switch (stCenter_Mag)  // Statemachine to detect Center Passes Magnetic
  {
    case cmag_Idle:                //state  0
      AMidCenter_Mag = 512;
      avMidCenter = AMidCenter_Mag << 4; 
      PositionCounter_Mag = 0;
      if (Drive_SyncMode == Drive_SyncByCenter_Mag) HaveSync = false;
      stCenter_Mag = cmag_DontLook; 
      if (ShowStates) Serial.println (stCenter_Mag);
    break;
    
    case cmag_DontLook:            // state 1
    if (PositionCounter_Mag >= TStartLookForCenter_Mag)
      {
        Track = 0; // prepare for track rising
        stCenter_Mag = cmag_TrackRising; 
        if (ShowStates) Serial.println (stCenter_Mag);
      }
    break;
   
    case cmag_TrackRising:         // state 2
      if (Adc_Center_Mag > Track) Track = Adc_Center_Mag;
      if (Adc_Center_Mag < (Track - 5)) // we are over the top
      {
        APeakCenter_Mag = Track; // freeze peak value
        stCenter_Mag = cmag_WaitCenter;
        if (ShowStates) Serial.println (stCenter_Mag);
      }
    break;
     
    case cmag_WaitCenter:          // state 3
      if (Adc_Center_Mag <= AMidCenter_Mag) // Found Center Pass
      {
        SeenCenter_Mag = true;
        TPassCenter_Mag = PositionCounter_Mag;
        TQuarterSwing = PositionCounter_Mag >> 1; // div 2
        if (Drive_SyncMode == Drive_SyncByCenter_Mag)
        {
          PositionCounter_Mag = 0;
          if (RimSyncMode == RimSyncByCenter_Mag) PositionCounter_Rim = 0;
          if (Drive_SyncMode == Drive_SyncByCenter_Mag) PositionCounter_Drive = 0;
          Center_North = Adc_North; // freeze PMS data at CenterPass
          Center_South = Adc_South;
          Center_West  = Adc_West;
          Center_East  = Adc_East;
          HaveSync = true;
          HalfSwing = !HalfSwing;
          SeenCenterLedCounter = 2000;
          RIMLED_1_OFF
          RIMLED_2_OFF
        }
        DIAGPIN_A14_T
        //Serial.println ("Found Center");
        stCenter_Mag = cmag_WaitSampleMidCenter;
        if (ShowStates) Serial.println (stCenter_Mag);
      }
    break;
    
    case cmag_WaitSampleMidCenter: // state 4
    if (PositionCounter_Mag > TQuarterSwing)
    {
      MidCenterSamples = 0;
      stCenter_Mag = cmag_SampleMidCenter;
      if (ShowStates) Serial.println (stCenter_Mag);
    }
    break;
    
    case cmag_SampleMidCenter:     // state 5
      // apply leaking bucket averaging with factor 1/16
      avMidCenter = avMidCenter - (avMidCenter >> 4) + Adc_Center_Mag;
      AMidCenter_Mag = avMidCenter >> 4;
      if (++MidCenterSamples >= 20) // average 20 samples
      {
        stCenter_Mag = cmag_DontLook;  // start over
        if (ShowStates) Serial.println (stCenter_Mag);
      }
    break;

    default: stCenter_Mag = cmag_Idle;
  } // switch
}

//********************************************************************************************************
// Statemachine to detect RimCoil Passes
// Called at 20 kHz rate from Timer1 interrupt handler
void DetectRim_Mag (void) 
{
  enum tStateRim_Mag {rmag_Idle = 0, rmag_DontLook, 
                      rmag_TrackRising, 
                      rmag_StartLook2, rmag_TrackFalling, 
                      rmag_WaitSampleMidRim, rmag_SampleMidRim};

  static tStateRim_Mag stRim_Mag;
  static long avTPassRim1_Mag;
  static int Track, avMidRim;
  static byte MidRimSamples;
  
  bool ShowStates = Opt2; // for diagnose

  PositionCounter_Rim++;
   
  switch (stRim_Mag) // Statemachine to detect Rim Passes Magnetic
  {
    case rmag_Idle:                   // state 0
      AMidRim_Mag = 512;
      avMidRim = AMidRim_Mag << 4; 
      stRim_Mag = rmag_DontLook; 
      if (ShowStates) Serial.println (stRim_Mag);
    break;

    case rmag_DontLook:               // state 1
      if (PositionCounter_Rim >= TStartLookForRim1_Mag)
      {
        Track = 0; // prepare for track rising
        stRim_Mag = rmag_TrackRising;
        if (ShowStates) Serial.println (stRim_Mag);
      }
    break;
  
    case rmag_TrackRising:            // state 2
      if ((Adc_Rim_Mag > 1022) || (Adc_Rim_Mag > 1022)) OverRangeRim1_Mag = true;
      if (Adc_Rim_Mag > Track) Track = Adc_Rim_Mag;
      if (Adc_Rim_Mag < (Track - 5)) // we are over the top
      {
        SeenRim1_Mag = true;
        APeakRim1_Mag = Track; // freeze the peakvalue
        TPassRim1_Mag = PositionCounter_Rim;
        if (AmplitudeControlMode == AmplitudeByRim_Mag)
        {
          // Leaking Bucket Average
          avTPassRim1_Mag = avTPassRim1_Mag - (avTPassRim1_Mag >> 4) + TPassRim1_Mag;
          MaxDrive = ((avTPassRim1_Mag >> 4) > SetPoint_Amplitude_Ticks);
          sprintf (PrintStr, "avTPassRim1_Mag: %5d Drive: ", avTPassRim1_Mag >> 4);
          Serial.print (PrintStr); Serial.println (MaxDrive ? "Max" : "Min");
        }  
        DIAGPIN_A13_H
        RIMLED_1_ON  // off at Center Pass
        stRim_Mag = rmag_StartLook2;
        if (ShowStates) Serial.println (stRim_Mag);
      }
      else if (PositionCounter_Rim > TMissedRim1_Mag)
        MissedRim1_Mag = true; // and we just go on
    break;

    case rmag_StartLook2:              // state 3
      if (PositionCounter_Rim >= TStartLookForRim2_Mag)
      {
        Track = 1023; // prepare for track falling
        stRim_Mag = rmag_TrackFalling;
        if (ShowStates) Serial.println (stRim_Mag);
      }
    break;

    case rmag_TrackFalling:             // state 4
      if ((Adc_Rim_Mag > 1022) || (Adc_Rim_Mag > 1022)) OverRangeRim2_Mag = true;
      if (Adc_Rim_Mag < Track) Track = Adc_Rim_Mag;
      if (Adc_Rim_Mag > (Track + 5)) // we are past the dip
      {
        SeenRim2_Mag = true;
        APeakRim2_Mag = Track; // freeze the dip value
        TPassRim2_Mag = PositionCounter_Rim;
        DIAGPIN_A13_L
        RIMLED_2_ON  // off at Center Pass
        stRim_Mag = rmag_WaitSampleMidRim;
        if (ShowStates) Serial.println (stRim_Mag);
      }
      else if (PositionCounter_Rim > TMissedRim2_Mag)
        MissedRim2_Mag = true; // and we just go on
    break;

    case rmag_WaitSampleMidRim:         // state 5
      // wait until the bob Just passed the center
      if (PositionCounter_Rim < 5) 
      {
        MidRimSamples = 0;
        stRim_Mag = rmag_SampleMidRim;
        if (ShowStates) Serial.println (stRim_Mag);
      }
    break;

    case rmag_SampleMidRim:              // state 6
      // apply leaking bucket averaging with factor 1/16
      avMidRim = avMidRim - (avMidRim >> 4) + Adc_Rim_Mag;
      AMidRim_Mag = avMidRim >> 4;
      if (++MidRimSamples >= 20) // average 20 samples
      {
        stRim_Mag = rmag_DontLook;  // start over
        if (ShowStates) Serial.println (stRim_Mag);
      }
    break;
  
    default: stRim_Mag = rmag_Idle;
  }
}

//********************************************************************************************************
// This is the workhorse in the system.
ISR (TIMER1_COMPA_vect) // interrupt at 20 kHz. 50 usec interval. Execution takes < 40 usec
{
  DIAGPIN_A15_H // Show frequency and duration of interrupt handler on pin A15
  
  static unsigned int OneSecondCounter, OneMinuteCounter, CommLedBlinker;
  static byte Channel;
  static bool CommOK;

  // Check Charron Ring:
  Touch_Charron = TOUCH_CHARRON;
  //if (Touch_Charron) Serial.println ("Touch Charron");
  
  // **************************** Leds ********************
  if (!CommOK) // Blue Commled blinks fast, otherwise slow
    if (++CommLedBlinker > 2000) {COMMLED_TOGGLE; CommLedBlinker = 0;} 
  if (DriveLedCounter > 0) {DRIVELED_ON; DriveLedCounter--;} else DRIVELED_OFF;
  if (SeenCenterLedCounter > 0) {CENTERLED_ON; SeenCenterLedCounter--;} else CENTERLED_OFF;
  if (HalfSwing) HALFSWINGLED_ON else HALFSWINGLED_OFF; 
   
  if (++OneSecondCounter >= 19999) // div by 20000 for 1 Hz
  {
    OneSecondCounter = 0;
    ReadOptionJumpers ();
    DoReadBME = true; 
    CommOK = HaveComm; // check that Communication is stil running
    HaveComm = false;
  }  

  // Update analog inputs
  // In each TimeSlice we get the result of the previously started conversion
  // Analog channels 0..7 are sequentially sampled at 20 kHz
  // so each channel at a 2.5 kHz rate
  switch (Channel) 
  {
    case 0: Adc_Rim_Mag = ADC; break; // result from channel 7
    case 1: Adc_North =   ADC; break; // result from channel 0
    case 2: Adc_South =   ADC; break; // result from channel 1
    case 3: Adc_West =    ADC; break; // result from channel 2
    case 4: Adc_East =    ADC; break; // result from channel 3
    case 5: 
      Adc_Center_Cap = ADC; // result from channel 4
      if ((Adc_Center_Cap > 1022) || (Adc_Center_Cap < 2)) OverRangeCenter_Cap = true; 
    break; 
    case 6:
      Adc_Center_Mag = ADC; // result from channel 5
      if ((Adc_Center_Mag > 1022) || (Adc_Center_Mag < 2)) OverRangeCenter_Mag = true; 
    break; 
    case 7: Adc_Rim_Cap =    ADC; break; // result from channel 6
  } 
  
  ADMUX = Channel++ | 0x40; // No ADLAR for 10 bit conversions, Ref = AVCC.
  Channel &= 0x07; // cycle 0..7 
  ADCSRA |= 0x40;  // start new A/D conversion

  if (EnableDetectorCenterPass_Mag) DetectCenter_Mag (); // all together takes < 40usec out of 50usec interrupt interval.
  if (EnableDetectorCenterPass_Cap) DetectCenter_Cap (); 
  if (EnableDetectorRimPass_Mag)  DetectRim_Mag ();

  // ************************** Handle Min / Max Drive current *******************
   
  if (ForceMaximalDriveCurrent) MaxDrive = true;
  if (ForceMinimalDriveCurrent) MaxDrive = false;
  if (MaxDrive)
  {
    UsingMaximalDriveCurrent = true;
    UsingMinimalDriveCurrent = false;
    OCR4A = Drive_MaximalCurrent; // set PWM
  }
  else
  {
    UsingMaximalDriveCurrent = false;
    UsingMinimalDriveCurrent = true;
    OCR4A = Drive_MinimalCurrent; // set PWM
  }
  if (!EnableDrive) UsingMaximalDriveCurrent =  UsingMinimalDriveCurrent = false; 
 
  // ***********************  Drive Pulse Control ******************************
  PositionCounter_Drive++;
     
  // Magnetic CenterPass detection is influenced by the Drive Pulse
  // Therefore Amplifier Gate is closed just before Drive Pulse and opened long after 
  if (PositionCounter_Drive == TDrive_Start - 5) {CENTER_GATE_DISCONNECT; CENTER_SHORT_SHORT} 
  if (PositionCounter_Drive == TDrive_Start) if (EnableDrive) {DRIVE_ON; DriveLedCounter = 5000;}
  if (PositionCounter_Drive == TDrive_Stop) DRIVE_OFF 
  if (PositionCounter_Drive == TDrive_Stop + 500) {CENTER_GATE_CONNECT; CENTER_SHORT_PASS}
 
  DIAGPIN_A15_L  // end of Timer1 interrupt handler
}

//********************************************************************************************************
void Init_BobControl (void)
{
  // Configure 16 bit Timer 1 for 20 kHz interrupts = 100 usec interval time
  TCCR1A = 0x00; // WaveformMode 4 = CTC
  TCCR1B = 0x09; // No prescaler, WaveformMode 4 = CTC, internal clock 16 MHz
  OCR1A =   799; // Div by 800 for 10 kHz
  TIMSK1 = 0x02; // Interrupt on OCRA match

  // A/D converter runs on 1 MHz clock
  // 1 conversion takes < 25 clocks or  < 25 usec
  ADCSRA = 0x84; // Enable A/D converter, clockdiv 16 for 1 MHz A/D clock
  ADCSRB = 0;    // channels 0..7 only
  ADMUX = 0x40;  // Ref = AVCC, No ADLAR for 10-bit conversion result
  DIDR0 = 0xFF;  // Disable digital function of analog pins A0 .. A7

  EnableDrive = true;
}
