unit u_logging;

// Purpose: Handles all logging

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

procedure AddToEventLog (EventMessage: shortstring);
procedure AddToLog (LogString: shortstring);
procedure LogIDs;
procedure LogData;
procedure LogRawHallData;
procedure ListData;   // to memo on desktop window
procedure WriteParametersToEventLog;
procedure NewLogFileNames;

var
  ExeDir, LoggingPath, LogFileName, EventLogFileName, HallDataLogFileName, LogStr: shortstring;

implementation

uses u_main, u_messages, u_calc, u_timestrings;

procedure AddToEventLog (EventMessage: shortstring);
var
  FileName: string;
  EventFile: text;
begin
  FileName:= EventLogFileName;
  assign (EventFile, FileName);
  if FileExists (FileName) then append (EventFile) else rewrite (EventFile);
  writeln (EventFile, DateTimeStamp + ': ' + EventMessage);
  close (EventFile);
end;

procedure AddToLog (LogString: shortstring);
var
  FileName: string;
  LogFile: text;
begin
  Filename:= LogFileName;
  assign (logFile, Filename);
  if FileExists (FileName) then append (LogFile) else rewrite (LogFile);
  writeln (LogFile, DateTimeStamp + LogString);
  close (LogFile);
end;

procedure AddToHallDataLog (LogString: shortstring);
var
  FileName: string;
  LogFile: text;
begin
  Filename:= HallDataLogFileName;
  assign (logFile, Filename);
  if FileExists (FileName) then append (LogFile) else rewrite (LogFile);
  writeln (LogFile, DateTimeStamp + LogString);
  close (LogFile);
end;

procedure LogRawHallData; // Called by TimerSixMinutesTimer
begin
  LogStr:= format (' *, %3d, %3d, %3d, %3d', [AdcHallNorth, AdcHallSouth, AdcHallEast,AdcHallWest]);
  AddToHallDataLog (LogStr);
end;

procedure LogIDs;
begin
  LogStr:= '**, Stat,   TPCenter,   TDiff, APC,    TPRim1, APR1,  TPRim2,  APR2,  ARim,  Short, Angle, T-Top, AmplX, AmplY, AnglX, AnglY, Temp, Hygro,  Baro,';
  AddToLog (LogStr);
end;

procedure LogData;
// Because of problems with locale and comma separated files we do not
// log floating point numbers. In stead integers with the values 10, 100, or 1000 fold.
begin
  LogStr:=
    format (' ,%6d, ', [Status]) +
    format ('%8d, ', [TPassCenter]) +
    format ('%8d, ', [TPasscenterDiff]) +
    format ('%4d, ', [APeakCenter]) +
    format ('%8d, ', [TPassRim_1]) +
    format ('%4d, ', [APeakRim_1]) +
    format ('%8d, ', [TPassRim_2]) +
    format ('%4d, ', [APeakRim_2]) +
    format ('%6d, ', [round (AmplitudeFromRim_mm * 10)]) +
    format ('%5d, ', [round (EllipseShortAxis_mm)]) +
    format ('%5d, ', [round (PrecessionAngle_Degrees)]) +
    format ('%5d, ', [TemperatureTop]) +
    format ('%5d, ', [round (AmplitudeX * 10000)]) +
    format ('%5d, ', [round (AmplitudeY * 10000)]) +
    format ('%5d, ', [round (AngleX * 180 / pi)]) +
    format ('%5d, ', [round (AngleY * 180 / pi)]) +
    format ('%5d, ', [TemperatureBME]) +
    format ('%5d, ', [HygroBME]) +
    format ('%5d, ', [BaroBME]);
  ;
  if MadeCompassScreenShot then LogStr += 'ScreenShot,';
  MadeCompassScreenShot:= false;
  AddToLog (LogStr);
end;

procedure ListData; // to memo on desktop window
begin
  if ShowOnMemo then MemoAdd (format (' %5d %5d ', [TPassCenter, TPassCenterDiff]));
end;

function BoolStr (V: boolean): string;
begin
  if V then BoolStr:= 'TRUE' else BoolStr:= 'False';
end;

procedure WriteParametersToEventLog;
begin
  AddToEventLog ('Changed Settings');
  AddToEventLog ('TStartLookCenter:      ' + format('%6d',[TStartLookForCenter]));
  AddToEventLog ('TMissedCenter:         ' + format('%6d',[TMissedCenter]));
  AddToEventLog ('TStartLookRim_1:       ' + format('%6d',[TStartLookForRim_1]));
  AddToEventLog ('TMissedRim_1:          ' + format('%6d',[TMissedRim_1]));
  AddToEventLog ('TStartLookRim_2:       ' + format('%6d',[TStartLookForRim_2]));
  AddToEventLog ('TMissedRim_2:          ' + format('%6d',[TMissedRim_2]));
  AddToEventLog ('RimCoilRadius:         ' + format('%4d',[RimCoilRadius_mm]));
  AddToEventLog ('SetPoint_Amplitude_mm: ' + format('%4d',[SetPoint_Amplitude_mm]));
  AddToEventLog ('TMidDrive:             ' + format('%6d',[TMidDrive]));
  AddToEventLog ('MaxDriveWidth:         ' + format('%4d',[TMaxDriveWidth]));
  AddToEventLog ('MinDriveWidth:         ' + format('%4d',[TMinDriveWidth]));
  AddToEventLog ('****************************************');
end;

procedure NewLogFileNames;
begin
  LogFileName:= DateTimeStamp2;
  FrmMain.TbxLogFileName.text:= LogFileName;
  EventLogFileName:= LogFileName + '.event';
  HallDataLogFilename:= LogFileName + '.hall';
  LogFileName:= LogFileName + '.log';
  AddToLog ('** '+ VERSIONSTRING);
  AddToEventLog ('** '+ VERSIONSTRING);
  AddToHallDataLog ('** '+ VERSIONSTRING);
  LogIDs;
  WriteParametersToEventLog;
end;

begin
// no initialisation code
end.

