/////////////////////////////////////////////////////////////////////////////////////////
// Module Name: Globals.h
// Project: Foucault Pendulum, Subsystem BobControl
// Target: Arduino MEGA + ETHernet Shield
// Last Mod: See .ino file
// jan@breem.nl,  janbee@hack42.nl, mail@foucaultpendulum.nl
// www.foucaultslinger.nl   www.foucaultpendulum.nl
//
// Pin Use (note: interface with Daisies and RPI is for future use)
// Pin 0, 1  Serial0: Programming, Monitor
// Pin 2  Signal from Daisies (also Pin 15)
// Pin 3  Free from Daisies
// Pin 6  Data In from Daisies
// Pin 7  Data Out to Daisies
// Pin 8  System Reset (via delay circuit)

// Pin 15 Signal from Daisies (also pin 2)
// Pin 16 TX2 to RPI
// Pin 17 RX2 from RPI
// Pin 18 RPI
// Pin 19 RPI
// Pin 20 SDA I2C interface
// Pin 21 SCL I2C interface

// Pin 22 PORTA 0 Hold Power ON
// note: Jumperblock is currently not used
// Pin 23 PORTA 1 Jumper
// Pin 25 PORTA 3 Jumper
// Pin 27 PORTA 5 Jumper
// Pin 29 PORTA 7 Jumper

// Pin 30 PORTC 7 Dallas Temperature Sensor(s)

// Pin 45 PORT  DrivePulse
// Pin 47 PORT  Led RIM  crossing
// Pin 49 PORT  Led Zero crossing

// anaog inputs
// A0  Hall North
// A1  Hall South
// A2  Hall East
// A3  Hall West
// A4  CenterPass
// A5  RimPass
// A6  Meas +24V Input
// A7 not used
// A8 through 15 are diagnostic outputs. see .ino file
/////////////////////////////////////////////////////////////////////////////////////////

#ifndef GLOBALS_H
#define GLOBALS_H

#include <Arduino.h>
#include <Wire.h>             // for I2C interface

// for Power Management
extern int BatteryVoltage_mV, BatteryCurrent_mA;
extern bool DoReadBatteryStatus;
  
// General
extern bool DidResetMyself, DidReSyncMyself, DoReportParameters, GeneralError;
void ResetArduino (byte);

// for Communication
extern bool SeenCommunication; // Detect communication failure
extern bool DoReportSettings;  // Flag to list settings
void Init_Messages (void);
void Update_Messages (void);

// For Storage
#define EEPROM_STARTADDRESS_PARAMETERS 100
#define RESET_FLAG_ADDRESS 1000
extern bool DoWriteParametersToEEprom;

// For Temperture measurement etc.
extern bool ReadaDallasSensor;
extern bool OneSecondPassed;
extern byte TopTemperature;

extern int TemperatureBME, HygroBME, BaroBME;
extern bool DoReadBME;

// for InMessage:
extern unsigned int 
  Command,
  TStartLookForCenter, TMissedCenter,
  TStartLookForRim_1,  TMissedRim_1,
  TStartLookForRim_2,  TMissedRim_2,
  TMidDrive,  SetPoint_Amplitude_mm, RimCoilRadius_mm;
extern byte TMaximalDriveWidth, TMinimalDriveWidth;  

// for OutMessage:
extern unsigned int 
  VersionNumber, Status, 
  TPassCenter, TPassRim_1, TPassRim_2;
extern byte 
  MessageNumber,
  AdcCenter, AdcRim,  
  APeakCenter, APeakRim_1, APeakRim_2,
  AdcHallNorth, AdcHallSouth, AdcHallEast, AdcHallWest,
  AdcV24;
extern int BatteryCurrentRaw;    

// for Bob Synchronisation:
enum t_State 
  {st_Prepare = 0, 
   st_LookForCenterApproach, 
   st_LookForCenterCross, 
   st_MissedCenter, 
   st_DontLookNow,
   st_LookForRim_1, 
   st_WaitRim_2, 
   st_LookForRim_2, 
   st_WaitToRepeatCycle};  
extern t_State State;  // for statemachine detecting Bob Passes
extern unsigned int PositionCounter, TPassCenter, TPassRim_1, TPassRim_2;
extern bool 
  ForceReSync, HaveSync, HalfSwing; 
extern bool SeenCenter, MissedCenter, 
  SeenRim_1, MissedRim_1, SeenRim_2, MissedRim_2;

// for Amplitude Control:
extern unsigned int 
  AmplitudeFromRim_mm, SetPoint_Amplitude_mm,
  TStartDrive, TStopDrive; 
extern bool 
  EnableDrive, DoUpdateAmplitudeControl,
  UseMaximalDriveWidth, UseMinimalDriveWidth, 
  ForceMaximalDriveWidth, ForceMinimalDriveWidth;
void UpdateAmplitudeControl (void); 
void UdateDrivePulses (bool Max);

// Forwards
void Init_BobControl (void);
void ReportSettings (void);     

// Output definitions
// Drive is PORTL bit 4, pin 45 
#define DriveON PORTL |= 0x10; 
#define DriveOFF PORTL &= ~0x10; 

// Zero led is on PORTL bit 0, pin 49
#define ZeroLedON PORTL |= 0x01; 
#define ZeroLedOFF PORTL &= ~0x01; 
#define ZeroLedTOGGLE PINL = 0x01; 

// Rim led is on PORTL bit 2, pin 47
#define RIMLEDON PORTL |= 0x04; 
#define RIMLEDOFF PORTL &= ~0x04; 
#define RIMLEDTOGGLE PINL = 0x04; 

// Switch to keep system switched on is on PORTA bit 2, pin 24
#define HOLDPOWER_ON  PORTA |=  0x01;
#define HOLDPOWER_OFF PORTA &= ~0x01;

// Switch for charging Battery is on PORTA bit 0, pin 22
#define CHARGEBATTERY_ON  PORTA |=  0x04;
#define CHARGEBATTERY_OFF PORTA &= ~0x04;

// Diagnostic pins Arduino MEGA, ADCH
#define USEDIAGPIN_A8 DDRK |= 0x01;
#define DIAGPIN_A8_H PORTK |= 0x01; 
#define DIAGPIN_A8_L PORTK &= ~0x01; 
#define DIAGPIN_A8_T PINK = 0x01;

#define USEDIAGPIN_A9 DDRK |= 0x02;
#define DIAGPIN_A9_H PORTK |= 0x02; 
#define DIAGPIN_A9_L PORTK &= ~0x02; 
#define DIAGPIN_A9_T PINK = 0x02;

#define USEDIAGPIN_A10 DDRK |= 0x04; 
#define DIAGPIN_A10_H PORTK |= 0x04; 
#define DIAGPIN_A10_L PORTK &= ~0x04; 
#define DIAGPIN_A10_T PINK = 0x04;

#define USEDIAGPIN_A11 DDRK |= 0x08; 
#define DIAGPIN_A11_H PORTK |= 0x08; 
#define DIAGPIN_A11_L PORTK &= ~0x08; 
#define DIAGPIN_A11_T PINK = 0x08;

#define USEDIAGPIN_A12 DDRK |= 0x10; 
#define DIAGPIN_A12_H PORTK |= 0x10; 
#define DIAGPIN_A12_L PORTK &= ~0x10; 
#define DIAGPIN_A12_T PINK = 0x10;

#define USEDIAGPIN_A13 DDRK |= 0x20; 
#define DIAGPIN_A13_H PORTK |= 0x20; 
#define DIAGPIN_A13_L PORTK &= ~0x20; 
#define DIAGPIN_A13_T PINK = 0x20;

#define USEDIAGPIN_A14 DDRK |= 0x40; 
#define DIAGPIN_A14_H PORTK |= 0x40; 
#define DIAGPIN_A14_L PORTK &= ~0x40; 
#define DIAGPIN_A14_T PINK = 0x40;

#define USEDIAGPIN_A15 DDRK |= 0x80; 
#define DIAGPIN_A15_H PORTK |= 0x80; 
#define DIAGPIN_A15_L PORTK &= ~0x80; 
#define DIAGPIN_A15_T PINK = 0x80;
#endif
